<?php
$albumName = $_GET['name'] ?? '';
$albumPath = "albums/" . basename($albumName);

if (!is_dir($albumPath)) {
  http_response_code(404);
  die("Album not found.");
}

// Find images (case-insensitive extensions)
$images = glob("$albumPath/*.{jpg,jpeg,png,gif,webp,JPG,JPEG,PNG,GIF,WEBP}", GLOB_BRACE);
// Normalize and keep only files
$images = array_values(array_filter($images, 'is_file'));
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title><?php echo htmlspecialchars(ucfirst($albumName)); ?> - Album</title>

<!-- Masonry (vanilla pkgd) -->
<script src="https://unpkg.com/masonry-layout@4/dist/masonry.pkgd.min.js"></script>

<style>
  :root{ --bg:#fafafa; --card:#fff; --muted:#666; }
  *{box-sizing:border-box}
  body{font-family:Inter, system-ui, -apple-system, 'Segoe UI', Roboto, 'Helvetica Neue', Arial; background:var(--bg); margin:0; padding:20px; color:#111}
  a.back{display:inline-block; margin-bottom:18px; text-decoration:none; color:var(--muted)}
  h1{margin:0 0 18px; text-align:center}

  /* Masonry grid */
  .grid{max-width:1400px; margin:0 auto}
  .grid-sizer{width:33.333%;}
  .grid-item{width:calc(33.333% - 18px); margin:9px 9px 18px 9px;}

  .thumb{
    display:block;
    border-radius:8px;
    overflow:hidden;
    background:var(--card);
    box-shadow:0 6px 18px rgba(17,17,17,0.06);
    cursor:pointer;
    transition:transform .15s ease, box-shadow .15s ease;
  }
  .thumb img{display:block; width:100%; height:auto}
  .thumb:focus,.thumb:hover{transform:translateY(-4px); box-shadow:0 12px 30px rgba(17,17,17,0.09)}

  /* Responsive */
  @media (max-width:900px){ .grid-sizer, .grid-item{width:50%} }
  @media (max-width:520px){ .grid-sizer, .grid-item{width:100%} }

  /* Overlay viewer (Instagram-style) */
  .overlay{position:fixed; inset:0; display:none; align-items:center; justify-content:center; background:rgba(0,0,0,0.85); z-index:9999; padding:20px; opacity:0; transition:opacity .25s ease}
  .overlay.open{display:flex; opacity:1}
  .viewer{position:relative; max-width:1200px; max-height:90vh; width:100%; display:flex; align-items:center; justify-content:center;}
  .viewer .imgwrap{flex:1 1 auto; display:flex; align-items:center; justify-content:center; position:relative;}
  .viewer img{max-width:100%; max-height:90vh; border-radius:8px; box-shadow:0 10px 30px rgba(0,0,0,0.6)}

  /* In-frame controls (left/right) */
  .in-btn{position:absolute; top:50%; transform:translateY(-50%); background:rgba(0,0,0,.35); border:0; color:#fff; font-size:26px; width:48px; height:48px; display:flex; align-items:center; justify-content:center; cursor:pointer; border-radius:50%; backdrop-filter:blur(4px); transition:background .2s}
  .in-btn:hover{background:rgba(0,0,0,.55)}
  #prevBtn{left:12px}
  #nextBtn{right:12px}

  /* Close button inside image */
  .in-close{position:absolute; right:10px; top:10px; z-index:10000; background:rgba(0,0,0,0.4); border-radius:50%; padding:6px; display:flex}
  .in-close button{background:transparent; border:0; color:#fff; font-size:20px; cursor:pointer}

  /* Caption */
  .caption{color:#ddd; text-align:center; margin-top:12px; font-size:14px}

  /* Small helper for older browsers */
  .hidden{display:none}

  /* Prevent scroll when open */
  body.no-scroll { overflow: hidden; }
</style>
</head>
<body>

<a href="index.php" class="back">← Back to albums</a>
<h1><?php echo htmlspecialchars(ucfirst($albumName)); ?></h1>

<div class="grid" id="gallery">
  <div class="grid-sizer"></div>
  <?php foreach ($images as $idx => $img): ?>
    <div class="grid-item">
      <div class="thumb" tabindex="0" data-index="<?php echo $idx; ?>" data-src="<?php echo htmlspecialchars($img); ?>" aria-label="Open image <?php echo htmlspecialchars(basename($img)); ?>">
        <img src="<?php echo htmlspecialchars($img); ?>" alt="<?php echo htmlspecialchars(basename($img)); ?>" loading="lazy">
      </div>
    </div>
  <?php endforeach; ?>
</div>

<!-- Overlay viewer -->
<div class="overlay" id="overlay" aria-hidden="true">
  <div class="viewer" role="dialog" aria-modal="true">
    <div class="imgwrap">
      <button id="prevBtn" class="in-btn" aria-label="Previous">◀</button>
      <img id="overlayImage" src="" alt="">
      <button id="nextBtn" class="in-btn" aria-label="Next">▶</button>
      <div class="in-close" id="inClose"><button aria-label="Close">✕</button></div>
    </div>
  </div>
  <div class="caption" id="overlayCaption"></div>
</div>

<script>
// Images array from PHP
const images = <?php echo json_encode(array_values($images), JSON_HEX_TAG|JSON_HEX_APOS|JSON_HEX_QUOT|JSON_HEX_AMP); ?>;

// Masonry init after images loaded
function initMasonry(){
  const grid = document.querySelector('.grid');
  if(!grid) return;
  const msnry = new Masonry(grid, {
    itemSelector: '.grid-item',
    columnWidth: '.grid-sizer',
    
    percentPosition: true,
    transitionDuration: '0.18s'
  });

  // Re-layout after each image loads (helps avoid overlap)
  const imgs = grid.querySelectorAll('img');
  imgs.forEach(img => {
    if(!img.complete) img.addEventListener('load', ()=> msnry.layout());
  });
  // Final layout pass
  setTimeout(()=> msnry.layout(), 150);
}

window.addEventListener('load', initMasonry);

// Overlay logic
const overlay = document.getElementById('overlay');
const overlayImage = document.getElementById('overlayImage');
const overlayCaption = document.getElementById('overlayCaption');
const prevBtn = document.getElementById('prevBtn');
const nextBtn = document.getElementById('nextBtn');
const inClose = document.getElementById('inClose');
let currentIndex = 0;

function openOverlay(index){
  if(images.length === 0) return;
  currentIndex = (index + images.length) % images.length;
  const src = images[currentIndex];
  overlayImage.src = src;
  overlayImage.alt = src.split('/').pop();
  overlayCaption.textContent = overlayImage.alt;
  overlay.classList.add('open');
  overlay.setAttribute('aria-hidden','false');
  document.body.classList.add('no-scroll');
}

function closeOverlay(){
  overlay.classList.remove('open');
  overlay.setAttribute('aria-hidden','true');
  overlayImage.src = '';
  document.body.classList.remove('no-scroll');
}

function showNext(){ openOverlay(currentIndex + 1); }
function showPrev(){ openOverlay(currentIndex - 1); }

// Click handlers for thumbnails
document.querySelectorAll('.thumb').forEach(el => {
  el.addEventListener('click', ()=>{
    openOverlay(parseInt(el.dataset.index,10));
  });
  el.addEventListener('keydown', (e)=>{
    if(e.key === 'Enter' || e.key === ' ') { e.preventDefault(); openOverlay(parseInt(el.dataset.index,10)); }
  });
});

// Close when clicking the dark backdrop (only when target === overlay)
overlay.addEventListener('click', (e)=>{
  if(e.target === overlay) closeOverlay();
});

// Close button inside image
inClose.addEventListener('click', closeOverlay);

// Nav
nextBtn.addEventListener('click', (e)=>{ e.stopPropagation(); showNext(); });
prevBtn.addEventListener('click', (e)=>{ e.stopPropagation(); showPrev(); });

// Swipe support
let touchStartX = 0;
overlay.addEventListener('touchstart', (e)=>{
  touchStartX = e.changedTouches[0].screenX;
});
overlay.addEventListener('touchend', (e)=>{
  const dx = e.changedTouches[0].screenX - touchStartX;
  if(Math.abs(dx) > 50){
    if(dx < 0) showNext(); else showPrev();
  }
});

// Keyboard
window.addEventListener('keydown', (e)=>{
  if(overlay.classList.contains('open')){
    if(e.key === 'Escape') closeOverlay();
    if(e.key === 'ArrowRight') showNext();
    if(e.key === 'ArrowLeft') showPrev();
  }
});

// Preload neighbor images for smooth navigation
overlayImage.addEventListener('load', ()=>{
  const next = new Image(); next.src = images[(currentIndex+1)%images.length];
  const prev = new Image(); prev.src = images[(currentIndex-1+images.length)%images.length];
});

// Accessibility: simple focus trap (keeps keyboard on overlay)
overlay.addEventListener('keydown', (e)=>{
  if(e.key === 'Tab') e.preventDefault();
});
</script>

</body>
</html>
